/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.readers;

import javax.xml.stream.XMLStreamException;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import eu.europa.ec.taxud.cesop.domain.XmlAddress;
import eu.europa.ec.taxud.cesop.domain.XmlCountryTypeAndValue;
import eu.europa.ec.taxud.cesop.domain.XmlDocSpec;
import eu.europa.ec.taxud.cesop.domain.XmlPsp;
import eu.europa.ec.taxud.cesop.domain.XmlReportedPayee;
import eu.europa.ec.taxud.cesop.domain.XmlTypeAndValue;
import eu.europa.ec.taxud.cesop.utils.ValidationConstants.KEY;
import eu.europa.ec.taxud.cesop.utils.ValidationConstants.XML;

import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.convertGreece;

/**
 * Utils class reading a ReportedPayee XML file.
 */
public class ReportedPayeeXmlReader {

    private final CesopXmlReader cesopXmlReader;
    private final XmlReportedPayee xmlReportedPayee = new XmlReportedPayee();

    private static final String UNSPECIFIED_ADDRESS_TYPE = "CESOP309";

    /**
     * Constructor.
     *
     * @param cesopXmlReader the XML reader.
     */
    public ReportedPayeeXmlReader(final CesopXmlReader cesopXmlReader) {
        this.cesopXmlReader = cesopXmlReader;
    }

    /**
     * Parse first part.
     *
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public void parseFirstPart() throws XMLStreamException {
        if (!this.cesopXmlReader.positionCursorOnStartElement(XML.REPORTED_PAYEE_QNAME)) {
            throw new XMLStreamException("Tag " + XML.REPORTED_PAYEE_QNAME + " not found.");
        }
        this.cesopXmlReader.getXmlStreamReaderWrapper().moveToNextElement();
        this.xmlReportedPayee.setNames(this.parseNames());
        this.xmlReportedPayee.setCountry(this.parseCountry());
        this.xmlReportedPayee.setAddresses(this.parseAddresses(xmlReportedPayee.getCountry()));
        this.xmlReportedPayee.setEmailAddresses(this.parseEmailAddresses());
        this.xmlReportedPayee.setWebPages(this.parseWebPages());
        this.xmlReportedPayee.setVatIdentifications(this.parseVatIdentifications());
        this.xmlReportedPayee.setTaxIdentifications(this.parseTaxIdentifications());
        this.xmlReportedPayee.setAccountIdentifiers(this.parseAccountIdentifiers());
    }

    /**
     * Parse second part.
     *
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public void parseSecondPart() throws XMLStreamException {
        this.xmlReportedPayee.setRepresentative(this.parseRepresentative());
        this.xmlReportedPayee.setDocSpec(this.parseDocSpec());
    }

    private List<XmlTypeAndValue> parseNames() throws XMLStreamException {
        final List<XmlTypeAndValue> names = new ArrayList<>();
        Optional<Map<String, String>> valuesMap;
        while ((valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.NAME_QNAME)).isPresent()) {
            names.add(new XmlTypeAndValue(
                    valuesMap.get().get(KEY.NAME_TYPE_KEY),
                    valuesMap.get().get(KEY.NAME_KEY),
                    valuesMap.get().get(KEY.NAME_OTHER_KEY)));
        }
        return names;
    }

    private String parseCountry() throws XMLStreamException {
        final Map<String, String> valuesMap = this.cesopXmlReader.readNextTagIntoMap(XML.COUNTRY_QNAME);
        return convertGreece(valuesMap.get(KEY.COUNTRY_KEY));
    }

    private List<XmlAddress> parseAddresses(String defaultCountry) throws XMLStreamException {
        final List<XmlAddress> addresses = new ArrayList<>();
        Optional<Map<String, String>> valuesMap;
        while ((valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.ADDRESS_QNAME)).isPresent()) {
            final XmlAddress xmlAddress = new XmlAddress();
            xmlAddress.setLegalAddressType(
                    valuesMap.get().get(KEY.ADDRESS_LEGAL_ADDRESS_TYPE_KEY) == null ? UNSPECIFIED_ADDRESS_TYPE : valuesMap.get().get(KEY.ADDRESS_LEGAL_ADDRESS_TYPE_KEY));
            xmlAddress.setCountryCode(
                    valuesMap.get().get(KEY.ADDRESS_COUNTRY_CODE_KEY) == null ? defaultCountry : convertGreece(valuesMap.get().get(KEY.ADDRESS_COUNTRY_CODE_KEY)));
            xmlAddress.setStreet(valuesMap.get().get(KEY.ADDRESS_STREET_KEY));
            xmlAddress.setBuildingIdentifier(valuesMap.get().get(KEY.ADDRESS_BUILDING_IDENTIFIER_KEY));
            xmlAddress.setSuiteIdentifier(valuesMap.get().get(KEY.ADDRESS_SUITE_IDENTIFIER_KEY));
            xmlAddress.setFloorIdentifier(valuesMap.get().get(KEY.ADDRESS_FLOOR_IDENTIFIER_KEY));
            xmlAddress.setDistrictName(valuesMap.get().get(KEY.ADDRESS_DISTRICT_NAME_KEY));
            xmlAddress.setPostOfficeBox(valuesMap.get().get(KEY.ADDRESS_POB_KEY));
            xmlAddress.setPostCode(valuesMap.get().get(KEY.ADDRESS_POST_CODE_KEY));
            xmlAddress.setCity(valuesMap.get().get(KEY.ADDRESS_CITY_KEY));
            xmlAddress.setCountrySubentity(valuesMap.get().get(KEY.ADDRESS_COUNTRY_SUBENTITY_KEY));
            xmlAddress.setAddressFree(valuesMap.get().get(KEY.ADDRESS_ADDRESS_FREE_KEY));
            addresses.add(xmlAddress);
        }
        return addresses;
    }

    private List<String> parseEmailAddresses() throws XMLStreamException {
        final List<String> emailAddresses = new ArrayList<>();
        Optional<Map<String, String>> valuesMap;
        while ((valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.EMAIL_ADDRESS_QNAME)).isPresent()) {
            emailAddresses.add(valuesMap.get().get(KEY.EMAIL_ADDRESS_KEY));
        }
        return emailAddresses;
    }

    private List<String> parseWebPages() throws XMLStreamException {
        final List<String> webPages = new ArrayList<>();
        Optional<Map<String, String>> valuesMap;
        while ((valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.WEBPAGE_QNAME)).isPresent()) {
            webPages.add(valuesMap.get().get(KEY.WEB_PAGE_KEY));
        }
        return webPages;
    }

    private List<XmlTypeAndValue> parseVatIdentifications() throws XMLStreamException {
        final List<XmlTypeAndValue> vatIdentifications = new ArrayList<>();
        if (this.cesopXmlReader.positionCursorOnStartElement(XML.TAX_IDENTIFICATION_QNAME)) {
            this.cesopXmlReader.getXmlStreamReaderWrapper().moveToNextElement();
            Optional<Map<String, String>> valuesMap;
            while ((valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.VAT_ID_QNAME)).isPresent()) {
                vatIdentifications.add(new XmlTypeAndValue(
                        convertGreece(valuesMap.get().get(KEY.VAT_ID_ISSUED_BY_KEY)),
                        valuesMap.get().get(KEY.VAT_ID_KEY)));
            }
        }
        return vatIdentifications;
    }

    private List<XmlCountryTypeAndValue> parseTaxIdentifications() throws XMLStreamException {
        final List<XmlCountryTypeAndValue> taxIdentifications = new ArrayList<>();
        Optional<Map<String, String>> valuesMap;
        while ((valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.TAX_ID_QNAME)).isPresent()) {
            taxIdentifications.add(new XmlCountryTypeAndValue(
                    convertGreece(valuesMap.get().get(KEY.TAX_ID_ISSUED_BY_KEY)),
                    valuesMap.get().get(KEY.TAX_ID_TYPE_KEY),
                    valuesMap.get().get(KEY.TAX_ID_KEY),
                    valuesMap.get().get(KEY.TAX_ID_OTHER_KEY)));
        }
        return taxIdentifications;
    }

    private List<XmlCountryTypeAndValue> parseAccountIdentifiers() throws XMLStreamException {
        final List<XmlCountryTypeAndValue> accountIdentifiers = new ArrayList<>();
        Optional<Map<String, String>> valuesMapOpt;
        while ((valuesMapOpt = this.cesopXmlReader.readNextTagIfEquals(XML.ACCOUNT_IDENTIFIER_QNAME)).isPresent()) {
            final Map<String, String> valuesMap = valuesMapOpt.get();
            accountIdentifiers.add(new XmlCountryTypeAndValue(
                    convertGreece(valuesMap.get(KEY.ACCOUNT_IDENTIFIER_COUNTRY_CODE_KEY)),
                    valuesMap.get(KEY.ACCOUNT_IDENTIFIER_TYPE_KEY),
                    valuesMap.get(KEY.ACCOUNT_IDENTIFIER_KEY),
                    valuesMap.get(KEY.ACCOUNT_IDENTIFIER_OTHER_KEY)));
        }
        return accountIdentifiers;
    }

    private XmlPsp parseRepresentative() throws XMLStreamException {
        if (this.cesopXmlReader.positionCursorOnStartElement() && this.cesopXmlReader.getStartElementName().equals(XML.REPRESENTATIVE_QNAME)) {
            this.cesopXmlReader.getXmlStreamReaderWrapper().moveToNextElement();
            final Optional<Map<String, String>> reportingPspMap = this.cesopXmlReader.readNextTagIfEquals(XML.REPRESENTATIVE_ID_QNAME);
            if (reportingPspMap.isPresent()) {
                final String pspIdType = reportingPspMap.get().get(KEY.REPRESENTATIVE_ID_TYPE_KEY);
                final String pspIdOther = reportingPspMap.get().get(KEY.REPRESENTATIVE_ID_OTHER_KEY);
                final String pspId = reportingPspMap.get().get(KEY.REPRESENTATIVE_ID_KEY);
                final XmlPsp xmlReportingPsp = new XmlPsp(pspIdType, pspId, pspIdOther);
                Optional<Map<String, String>> valuesMap;
                while ((valuesMap = this.cesopXmlReader.readNextTagIfEquals(XML.NAME_QNAME)).isPresent()) {
                    final String pspNameType = valuesMap.get().get(KEY.NAME_TYPE_KEY);
                    final String pspNameOther = valuesMap.get().get(KEY.NAME_OTHER_KEY);
                    final String pspName = valuesMap.get().get(KEY.NAME_KEY);
                    xmlReportingPsp.addName(pspNameType, pspName, pspNameOther);
                }
                return xmlReportingPsp;
            }

        }
        return null;
    }

    private XmlDocSpec parseDocSpec() throws XMLStreamException {
        final Map<String, String> valuesMap = this.cesopXmlReader.readNextTagIntoMap(XML.DOC_SPEC_QNAME);
        final XmlDocSpec xmlDocSpec = new XmlDocSpec();
        xmlDocSpec.setDocTypeIndic(valuesMap.get(KEY.DOC_SPEC_DOC_TYPE_INDIC_KEY));
        xmlDocSpec.setDocRefId(valuesMap.get(KEY.DOC_SPEC_DOC_REF_ID_KEY));
        xmlDocSpec.setCorrMessageRefId(valuesMap.get(KEY.DOC_SPEC_CORR_MESSAGE_REF_ID_KEY));
        xmlDocSpec.setCorrDocRefId(valuesMap.get(KEY.DOC_SPEC_CORR_DOC_REF_ID_KEY));
        return xmlDocSpec;
    }

    /**
     * Gets xml reported payee.
     *
     * @return the xml reported payee
     */
    public XmlReportedPayee getXmlReportedPayee() {
        return this.xmlReportedPayee;
    }
}
